(() => {
  /*
    Print Preflight Visualizer (Local) v2.1
    - Finds risky areas by geometry (not blur-diff):
      - Thin strokes (min stroke width)
      - Narrow gaps (min knockout gap)
    - Computes effective PPI from print size (mm) and image size (px)
    - Auto-binarizes ink mask (Otsu) and auto-detects ink polarity
    - EDT-based thickness estimation + local maxima sampling
    - Stress test: downsample to a target PPI then upscale back (worst-case approximation)

    Overlay colors:
      - Red: thin strokes
      - Orange: narrow gaps
  */

  const $ = (id) => document.getElementById(id);

  // UI
  const btnStd = $("btnStd");
  const btnStrong = $("btnStrong");

  // anchors + mm
  const btnAncTee = $("ancTee");
  const btnAncA4 = $("ancA4");
  const btnAncPost = $("ancPost");
  const btnAncCustom = $("ancCustom");
  const elMmW = $("mmW");
  const elMmH = $("mmH");
  const infoEl = $("info");

  // stress test + profiles + bg
  const btnStOff = $("stOff");
  const btnSt150 = $("st150");
  const btnSt200 = $("st200");
  const btnPfWhite = $("pfWhite");
  const btnPfBlack = $("pfBlack");
  const btnBgLight = $("bgLight");
  const btnBgDark = $("bgDark");
  const elHL = $("hl");
  const elFit = $("fit");

  const btnAnalyze = $("analyze");
  const btnClear = $("clear");
  const statusEl = $("status");

  const dropzone = $("dropzone");
  const fileInput = $("file");

  const cOrig = $("cOrig");
  const cRes = $("cRes");
  const cHl = $("cHl");

  // Hard fail if IDs missing (prevents half-broken runtime)
  const required = [
    ["btnStd", btnStd], ["btnStrong", btnStrong],
    ["ancTee", btnAncTee], ["ancA4", btnAncA4], ["ancPost", btnAncPost], ["ancCustom", btnAncCustom],
    ["mmW", elMmW], ["mmH", elMmH], ["info", infoEl],
    ["stOff", btnStOff], ["st150", btnSt150], ["st200", btnSt200],
    ["pfWhite", btnPfWhite], ["pfBlack", btnPfBlack],
    ["bgLight", btnBgLight], ["bgDark", btnBgDark],
    ["hl", elHL], ["fit", elFit],
    ["analyze", btnAnalyze], ["clear", btnClear], ["status", statusEl],
    ["dropzone", dropzone], ["file", fileInput],
    ["cOrig", cOrig], ["cRes", cRes], ["cHl", cHl],
  ];
  const missingIds = required.filter(([, el]) => !el).map(([id]) => id);
  if (missingIds.length) {
    const el = document.createElement("div");
    el.style.padding = "12px";
    el.style.color = "#ffb4b4";
    el.style.fontFamily = "system-ui, -apple-system, Segoe UI, Roboto, Arial";
    el.textContent = `Initialization failed (missing IDs: ${missingIds.join(", ")})`;
    document.body.prepend(el);
    console.error("Missing IDs:", missingIds);
    return;
  }

  const ctxOrig = cOrig.getContext("2d", { willReadFrequently: true });
  const ctxRes = cRes.getContext("2d", { willReadFrequently: true });
  const ctxHl = cHl.getContext("2d", { willReadFrequently: true });

  // Offscreen analysis canvas (stable sampling)
  const cWork = document.createElement("canvas");
  const ctxWork = cWork.getContext("2d", { willReadFrequently: true });

  // Stress test (downsample → upsample)
  const cStress = document.createElement("canvas");
  const ctxStress = cStress.getContext("2d");

  // For preview-bg auto detection (tiny matte sampling)
  const cMat = document.createElement("canvas");
  const ctxMat = cMat.getContext("2d", { willReadFrequently: true });

  const LS_BG_MODE = "print_preflight_preview_bg"; // "light" | "dark"
  const LS_BG_LOCK = "print_preflight_preview_bg_locked"; // "1" | "0"
  const LS_PRINT_PROFILE = "print_preflight_print_profile"; // "white" | "black"
  const LS_ANCHOR = "print_preflight_anchor"; // "tee" | "a4" | "post" | "custom"
  const LS_MM_W = "print_preflight_mm_w";
  const LS_MM_H = "print_preflight_mm_h";
  const LS_STRESS_PPI = "print_preflight_stress_ppi"; // "0" | "150" | "200"

  let previewBgMode = "dark";
  let previewBgLocked = false;

  let imgBitmap = null;
  let imgName = "image";
  let mode = "std"; // std | strong
  let printProfile = "white"; // white | black
  let anchor = "tee"; // tee | a4 | post | custom
  let stressPPI = 0; // 0 | 150 | 200

  // Cache for re-render
  let lastW = 0, lastH = 0;
  let lastOverlay = null; // ImageData
  let analyzed = false;

  // Performance cap
  const MAX_PIXELS = 3_000_000;

  // EDT discrete limitation safeguard
  const MIN_DIAMETER_PX_FOR_DETECTION = 2.2; // ensure radius >= 1.1px

  // Print profiles (mm thresholds are the core)
  const PRINT_PROFILES = {
    white: {
      label: "White garment (no underbase)",
      presets: {
        std: { minStrokeMm: 0.17, minGapMm: 0.17, dilatePx: 2 },
        strong: { minStrokeMm: 0.20, minGapMm: 0.20, dilatePx: 3 },
      }
    },
    black: {
      label: "Dark garment (with underbase)",
      presets: {
        std: { minStrokeMm: 0.20, minGapMm: 0.20, dilatePx: 2 },
        strong: { minStrokeMm: 0.30, minGapMm: 0.30, dilatePx: 3 },
      }
    }
  };

  // Anchors
  const ANCHORS = {
    tee: { label: "T-shirt 12×16", wMm: 304.8, hMm: 406.4, sugW: 4500, sugH: 6000, basePpi: 375 },
    a4: { label: "A4", wMm: 210, hMm: 297, sugW: 2480, sugH: 3508, basePpi: 300 },
    post: { label: "Postcard", wMm: 100, hMm: 148, sugW: 1181, sugH: 1748, basePpi: 300 },
    custom: { label: "Custom", wMm: null, hMm: null, sugW: null, sugH: null, basePpi: 300 }
  };

  function getPreset() {
    const prof = PRINT_PROFILES[printProfile] || PRINT_PROFILES.white;
    return prof.presets[mode] || prof.presets.std;
  }

  function setStatus(msg) { statusEl.textContent = msg; }

  function getPrintMm() {
    const w = parseFloat(elMmW.value);
    const h = parseFloat(elMmH.value);
    return { w, h, ok: Number.isFinite(w) && Number.isFinite(h) && w > 0 && h > 0 };
  }

  function updateAnalyzeEnabled() {
    const mm = getPrintMm();
    btnAnalyze.disabled = !imgBitmap || !mm.ok;
    btnClear.disabled = !imgBitmap;

    // download buttons: orig after load / result after analyze
    document.querySelectorAll("button.dl").forEach(b => {
      const t = b.dataset.target;
      if (t === "orig") b.disabled = !imgBitmap;
      if (t === "result") b.disabled = !imgBitmap || !analyzed;
    });
  }

  function applyPrintProfileClass() {
    btnPfWhite.classList.toggle("primary", printProfile === "white");
    btnPfBlack.classList.toggle("primary", printProfile === "black");
  }

  function setPrintProfile(p) {
    if (!PRINT_PROFILES[p]) p = "white";
    printProfile = p;
    applyPrintProfileClass();

    // persist
    try { localStorage.setItem(LS_PRINT_PROFILE, printProfile); } catch (_) { }

    // Lite UX: sync preview background to print profile
    setPreviewBg(printProfile === "black" ? "dark" : "light");

    updateInfoBar();

    if (imgBitmap) {
      analyzed = false;
      lastOverlay = null;
      drawOrigOnly();
      const label = (PRINT_PROFILES[printProfile] || PRINT_PROFILES.white).label;
      setStatus(`Switched print profile to “${label}”. (Preview background synced)`);
      updateAnalyzeEnabled();
    }
  }

  function loadPrintProfilePref() {
    try {
      const p = localStorage.getItem(LS_PRINT_PROFILE);
      if (p === "white" || p === "black") printProfile = p;
    } catch (_) {
      printProfile = "white";
    }
    applyPrintProfileClass();
  }

  // IMPORTANT: also apply matte to .viewer background (not only body theme)
  function applyPreviewBgClass() {
    const isLight = (previewBgMode === "light");

    // UI theme (optional)
    document.body.classList.toggle("mat-light", isLight);
    document.body.classList.toggle("mat-dark", !isLight);

    // Actual preview matte
    const matte = isLight ? "#ffffff" : "#0b0f16";
    document.querySelectorAll(".viewer").forEach((el) => {
      el.style.background = matte;
    });

    // segmented buttons
    btnBgLight.classList.toggle("primary", isLight);
    btnBgDark.classList.toggle("primary", !isLight);
  }

  function setPreviewBg(mode, lock) {
    if (mode !== "light" && mode !== "dark") return;
    previewBgMode = mode;
    if (typeof lock === "boolean") previewBgLocked = lock;

    applyPreviewBgClass();

    try {
      localStorage.setItem(LS_BG_MODE, previewBgMode);
      localStorage.setItem(LS_BG_LOCK, previewBgLocked ? "1" : "0");
    } catch (_) { }
  }

  function loadPreviewBgPref() {
    try {
      const m = localStorage.getItem(LS_BG_MODE);
      const l = localStorage.getItem(LS_BG_LOCK);
      if (m === "light" || m === "dark") previewBgMode = m;
      previewBgLocked = (l === "1");
    } catch (_) {
      previewBgMode = "dark";
      previewBgLocked = false;
    }
    applyPreviewBgClass();
  }

  function autoDetectPreviewBg(bitmap) {
    // Only if transparency exists significantly, detect by ink median luminance
    const S = 128;
    cMat.width = S;
    cMat.height = S;
    ctxMat.clearRect(0, 0, S, S);
    ctxMat.imageSmoothingEnabled = true;
    ctxMat.drawImage(bitmap, 0, 0, S, S);
    const data = ctxMat.getImageData(0, 0, S, S).data;
    const total = S * S;

    let transparent = 0;
    const lum = [];
    for (let i = 0; i < data.length; i += 4) {
      const a = data[i + 3];
      if (a === 0) { transparent++; continue; }
      if (a < 16) continue;
      const r = data[i], g = data[i + 1], b = data[i + 2];
      const l = 0.2126 * r + 0.7152 * g + 0.0722 * b;
      lum.push(l);
    }

    const tr = transparent / total;
    if (tr < 0.03) return "dark";
    if (lum.length < 64) return "dark";

    lum.sort((a, b) => a - b);
    const med = lum[(lum.length / 2) | 0];
    return (med < 130) ? "light" : "dark";
  }

  function applyAnchorClass() {
    btnAncTee.classList.toggle("primary", anchor === "tee");
    btnAncA4.classList.toggle("primary", anchor === "a4");
    btnAncPost.classList.toggle("primary", anchor === "post");
    btnAncCustom.classList.toggle("primary", anchor === "custom");
  }

  function setAnchor(a, persist = true, setMm = true) {
    if (!ANCHORS[a]) return;
    anchor = a;
    if (setMm && a !== "custom") {
      elMmW.value = String(ANCHORS[a].wMm);
      elMmH.value = String(ANCHORS[a].hMm);
    }
    applyAnchorClass();
    if (persist) {
      try { localStorage.setItem(LS_ANCHOR, anchor); } catch (_) { }
    }
    updateInfoBar();
    updateAnalyzeEnabled();
    if (imgBitmap) {
      analyzed = false;
      lastOverlay = null;
      drawOrigOnly();
      setStatus("Print size (mm) changed. Re-run Analyze if needed.");
    }
  }

  function applyStressClass() {
    btnStOff.classList.toggle("primary", stressPPI === 0);
    btnSt150.classList.toggle("primary", stressPPI === 150);
    btnSt200.classList.toggle("primary", stressPPI === 200);
  }

  function setStress(ppi, persist = true) {
    if (![0, 150, 200].includes(ppi)) return;
    stressPPI = ppi;
    applyStressClass();
    if (persist) {
      try { localStorage.setItem(LS_STRESS_PPI, String(stressPPI)); } catch (_) { }
    }
    updateInfoBar();
    if (imgBitmap) {
      analyzed = false;
      lastOverlay = null;
      drawOrigOnly();
      setStatus(stressPPI ? `Switched to stress test (${stressPPI}). Click Analyze to apply.` : "Stress test set to OFF. Click Analyze to apply.");
    }
  }

  function loadSpecPrefs() {
    try {
      const a = localStorage.getItem(LS_ANCHOR);
      if (a && ANCHORS[a]) anchor = a;
      const w = localStorage.getItem(LS_MM_W);
      const h = localStorage.getItem(LS_MM_H);
      if (w) elMmW.value = w;
      if (h) elMmH.value = h;
    } catch (_) { }

    try {
      const s = parseInt(localStorage.getItem(LS_STRESS_PPI) || "0", 10);
      if ([0, 150, 200].includes(s)) stressPPI = s;
    } catch (_) { stressPPI = 0; }

    applyAnchorClass();
    applyStressClass();
  }

  function persistMm() {
    const mm = getPrintMm();
    if (!mm.ok) return;
    try {
      localStorage.setItem(LS_MM_W, String(mm.w));
      localStorage.setItem(LS_MM_H, String(mm.h));
    } catch (_) { }
  }

  function fmtPct(x) { return `${(x * 100).toFixed(1)}%`; }

  function updateInfoBar() {
    const mm = getPrintMm();
    if (!mm.ok) {
      infoEl.textContent = "Enter print size (mm).";
      return;
    }

    const inchW = mm.w / 25.4;
    const inchH = mm.h / 25.4;
    const sug = ANCHORS[anchor] || ANCHORS.custom;
    const sugText = (sug.sugW && sug.sugH) ? `Suggested final px ${sug.sugW}×${sug.sugH}` : "Suggested final px (no anchor)";
    const stressText = stressPPI ? `Stress ${stressPPI}` : "Stress OFF";

    if (!imgBitmap) {
      infoEl.textContent = `${sugText} / ${stressText} / Print size ${mm.w}×${mm.h}mm`;
      return;
    }

    const ppiX = imgBitmap.width / inchW;
    const ppiY = imgBitmap.height / inchH;

    const rImg = imgBitmap.width / imgBitmap.height;
    const rMm = mm.w / mm.h;
    const drift = Math.abs((rImg / rMm) - 1);
    const driftText = drift <= 0.01 ? `Aspect OK (drift ${fmtPct(drift)})` : `Aspect NG (drift ${fmtPct(drift)}: match via crop or padding)`;

    infoEl.textContent = `${sugText} / ${stressText} / Effective PPI ${ppiX.toFixed(0)}×${ppiY.toFixed(0)} / ${driftText}`;
  }

  // ---------- initial state ----------
  btnStd.classList.add("primary");
  elHL.checked = true;
  elFit.checked = true;
  document.body.classList.remove("nofit");

  loadPrintProfilePref();
  loadPreviewBgPref();
  loadSpecPrefs();

  if (!["tee", "a4", "post", "custom"].includes(anchor)) {
    setAnchor("tee", false, true);
  } else {
    applyAnchorClass();
    updateInfoBar();
    updateAnalyzeEnabled();
  }

  setStatus("Drag & drop an image, or click to choose.");

  // ---------- mode buttons ----------
  btnStd.addEventListener("click", () => {
    mode = "std";
    btnStd.classList.add("primary");
    btnStrong.classList.remove("primary");
    if (imgBitmap) {
      analyzed = false;
      lastOverlay = null;
      drawOrigOnly();
      setStatus("Switched to Standard. Click Analyze to apply.");
    }
  });

  btnStrong.addEventListener("click", () => {
    mode = "strong";
    btnStrong.classList.add("primary");
    btnStd.classList.remove("primary");
    if (imgBitmap) {
      analyzed = false;
      lastOverlay = null;
      drawOrigOnly();
      setStatus("Switched to Stricter. Click Analyze to apply.");
    }
  });

  // anchors
  btnAncTee.addEventListener("click", () => setAnchor("tee"));
  btnAncA4.addEventListener("click", () => setAnchor("a4"));
  btnAncPost.addEventListener("click", () => setAnchor("post"));
  btnAncCustom.addEventListener("click", () => setAnchor("custom", true, false));

  // print mm
  ["input", "change"].forEach((ev) => {
    elMmW.addEventListener(ev, () => {
      persistMm();
      if (anchor !== "custom") {
        anchor = "custom";
        applyAnchorClass();
        try { localStorage.setItem(LS_ANCHOR, anchor); } catch (_) { }
      }
      updateInfoBar();
      updateAnalyzeEnabled();
      if (imgBitmap) {
        analyzed = false;
        lastOverlay = null;
        drawOrigOnly();
        setStatus("Print size (mm) changed. Re-run Analyze if needed.");
      }
    });

    elMmH.addEventListener(ev, () => {
      persistMm();
      if (anchor !== "custom") {
        anchor = "custom";
        applyAnchorClass();
        try { localStorage.setItem(LS_ANCHOR, anchor); } catch (_) { }
      }
      updateInfoBar();
      updateAnalyzeEnabled();
      if (imgBitmap) {
        analyzed = false;
        lastOverlay = null;
        drawOrigOnly();
        setStatus("Print size (mm) changed. Re-run Analyze if needed.");
      }
    });
  });

  // stress
  btnStOff.addEventListener("click", () => setStress(0));
  btnSt150.addEventListener("click", () => setStress(150));
  btnSt200.addEventListener("click", () => setStress(200));

  // print profile
  btnPfWhite.addEventListener("click", () => setPrintProfile("white"));
  btnPfBlack.addEventListener("click", () => setPrintProfile("black"));

  // preview background (manual lock/unlock)
  btnBgLight.addEventListener("click", () => {
    if (previewBgMode === "light" && previewBgLocked) {
      setPreviewBg("light", false);
      if (imgBitmap) setStatus("Preview background: light (back to Auto; next load will auto-detect)");
      return;
    }
    setPreviewBg("light", true);
    if (imgBitmap) setStatus("Preview background: light (fixed). Switch to dark if needed.");
  });

  btnBgDark.addEventListener("click", () => {
    if (previewBgMode === "dark" && previewBgLocked) {
      setPreviewBg("dark", false);
      if (imgBitmap) setStatus("Preview background: dark (back to Auto; next load will auto-detect)");
      return;
    }
    setPreviewBg("dark", true);
    if (imgBitmap) setStatus("Preview background: dark (fixed). Switch to light if needed.");
  });

  // highlight toggle
  elHL.addEventListener("change", () => {
    if (!imgBitmap) return;
    renderOverlay();
  });

  // fit toggle
  elFit.addEventListener("change", () => {
    if (elFit.checked) document.body.classList.remove("nofit");
    else document.body.classList.add("nofit");
  });

  // ---------- drag & drop ----------
  ["dragenter", "dragover", "dragleave", "drop"].forEach((type) => {
    window.addEventListener(type, (e) => { e.preventDefault(); }, { capture: true });
  });

  window.addEventListener("drop", (e) => {
    const path = (typeof e.composedPath === "function") ? e.composedPath() : [];
    if (path.includes(dropzone) || path.includes(fileInput)) return;
    const f = e.dataTransfer?.files?.[0];
    if (f) loadFile(f);
  }, { capture: true });

  dropzone.addEventListener("click", () => {
    fileInput.value = "";
    fileInput.click();
  });

  dropzone.addEventListener("dragenter", (e) => { e.preventDefault(); e.stopPropagation(); dropzone.classList.add("drag"); });
  dropzone.addEventListener("dragover", (e) => { e.preventDefault(); e.stopPropagation(); dropzone.classList.add("drag"); e.dataTransfer.dropEffect = "copy"; });
  dropzone.addEventListener("dragleave", (e) => { e.preventDefault(); e.stopPropagation(); dropzone.classList.remove("drag"); });

  dropzone.addEventListener("drop", (e) => {
    e.preventDefault(); e.stopPropagation();
    dropzone.classList.remove("drag");
    const f = e.dataTransfer?.files?.[0];
    if (f) loadFile(f);
  });

  fileInput.addEventListener("change", (e) => {
    const f = e.target.files?.[0];
    if (f) loadFile(f);
  });

  async function loadFile(file) {
    try {
      if (!file.type?.startsWith("image/")) {
        setStatus("Please choose an image (PNG recommended).");
        return;
      }

      setStatus("Loading image…");
      imgName = (file.name || "image").replace(/\.[^/.]+$/, "");
      imgBitmap = await createImageBitmap(file);

      // auto preview bg only when user didn't lock it
      if (!previewBgLocked) {
        const auto = autoDetectPreviewBg(imgBitmap);
        setPreviewBg(auto, false);
      }

      analyzed = false;
      updateInfoBar();
      updateAnalyzeEnabled();
      resetCache();
      drawOrigOnly();

      setStatus(`Loaded: ${imgBitmap.width}×${imgBitmap.height}px. The right pane stays blank until you click Analyze. Choose Standard/Stricter and click Analyze.`);
    } catch (err) {
      console.error(err);
      setStatus("Failed to load the image. Try exporting as PNG and reloading.");
      updateAnalyzeEnabled();
    }
  }

  function resetCache() {
    lastW = 0; lastH = 0;
    lastOverlay = null;
    ctxRes.clearRect(0, 0, cRes.width, cRes.height);
    ctxHl.clearRect(0, 0, cHl.width, cHl.height);
  }

  function setCanvasSizeAll(w, h) {
    [cOrig, cRes, cHl].forEach(c => { c.width = w; c.height = h; });
  }

  function chooseAnalysisScale(imgWpx, imgHpx, printWmm, printHmm, minFeatureMm) {
    const area = imgWpx * imgHpx;
    let s = (area <= MAX_PIXELS) ? 1 : Math.sqrt(MAX_PIXELS / area);
    s = Math.max(0.25, Math.min(1, s));

    if (Number.isFinite(printWmm) && Number.isFinite(printHmm) && printWmm > 0 && printHmm > 0) {
      const pxPerMm0 = Math.min(imgWpx / printWmm, imgHpx / printHmm);
      const minPx0 = Math.max(0.001, minFeatureMm * pxPerMm0);
      const need = MIN_DIAMETER_PX_FOR_DETECTION / minPx0;
      if (s < need) s = Math.min(1, need);
    }
    return s;
  }

  function drawOrigOnly() {
    if (!imgBitmap) return;

    const p = getPreset();
    const minFeatureMm = Math.min(p.minStrokeMm, p.minGapMm);
    const mm = getPrintMm();
    const scale = chooseAnalysisScale(imgBitmap.width, imgBitmap.height, mm.ok ? mm.w : NaN, mm.ok ? mm.h : NaN, minFeatureMm);
    const w = Math.max(1, Math.round(imgBitmap.width * scale));
    const h = Math.max(1, Math.round(imgBitmap.height * scale));

    setCanvasSizeAll(w, h);

    ctxOrig.clearRect(0, 0, w, h);
    ctxOrig.drawImage(imgBitmap, 0, 0, w, h);

    // right stays blank before analyze
    ctxRes.clearRect(0, 0, w, h);
    ctxHl.clearRect(0, 0, w, h);

    lastW = w; lastH = h;
    lastOverlay = null;
  }

  function drawForAnalysis() {
    if (!imgBitmap) return;

    const p = getPreset();
    const minFeatureMm = Math.min(p.minStrokeMm, p.minGapMm);
    const mm = getPrintMm();
    const scale = chooseAnalysisScale(imgBitmap.width, imgBitmap.height, mm.ok ? mm.w : NaN, mm.ok ? mm.h : NaN, minFeatureMm);
    const w = Math.max(1, Math.round(imgBitmap.width * scale));
    const h = Math.max(1, Math.round(imgBitmap.height * scale));

    setCanvasSizeAll(w, h);

    cWork.width = w;
    cWork.height = h;
    ctxWork.imageSmoothingEnabled = false;
    ctxWork.clearRect(0, 0, w, h);
    ctxWork.drawImage(imgBitmap, 0, 0, w, h);

    // stress test: reduce to target PPI then upscale back
    if (stressPPI && mm.ok) {
      const inchW = mm.w / 25.4;
      const inchH = mm.h / 25.4;
      const ppiX = w / inchW;
      const ppiY = h / inchH;
      const ppiEff = Math.min(ppiX, ppiY);
      const ratio = Math.min(1, stressPPI / ppiEff);
      if (ratio < 0.999) {
        const tw = Math.max(1, Math.round(w * ratio));
        const th = Math.max(1, Math.round(h * ratio));
        cStress.width = tw;
        cStress.height = th;
        ctxStress.imageSmoothingEnabled = true;
        ctxStress.clearRect(0, 0, tw, th);
        ctxStress.drawImage(cWork, 0, 0, tw, th);

        ctxWork.imageSmoothingEnabled = true;
        ctxWork.clearRect(0, 0, w, h);
        ctxWork.drawImage(cStress, 0, 0, w, h);
        ctxWork.imageSmoothingEnabled = false;
      }
    }

    // redraw
    ctxOrig.imageSmoothingEnabled = true;
    ctxRes.imageSmoothingEnabled = true;

    ctxOrig.clearRect(0, 0, w, h);
    ctxOrig.drawImage(imgBitmap, 0, 0, w, h);

    ctxRes.clearRect(0, 0, w, h);
    ctxRes.drawImage(imgBitmap, 0, 0, w, h);

    ctxHl.clearRect(0, 0, w, h);
  }

  // ---------- clear ----------
  btnClear.addEventListener("click", () => {
    imgBitmap = null;
    imgName = "image";
    analyzed = false;

    ctxOrig.clearRect(0, 0, cOrig.width, cOrig.height);
    ctxRes.clearRect(0, 0, cRes.width, cRes.height);
    ctxHl.clearRect(0, 0, cHl.width, cHl.height);

    fileInput.value = "";
    resetCache();
    updateAnalyzeEnabled();
    updateInfoBar();
    setStatus("Drag & drop an image, or click to choose.");
  });

  // ---------- analyze ----------
  btnAnalyze.addEventListener("click", () => {
    if (!imgBitmap) return;

    try {
      setStatus("Analyzing… (large images are automatically downscaled for analysis)");
      btnAnalyze.disabled = true;

      drawForAnalysis();

      const w = cRes.width, h = cRes.height;
      const imgData = ctxWork.getImageData(0, 0, w, h);

      const mm = getPrintMm();
      if (!mm.ok) {
        setStatus("Print size (mm) is missing.");
        btnAnalyze.disabled = false;
        return;
      }

      const pxPerMmX = w / mm.w;
      const pxPerMmY = h / mm.h;
      const pxPerMm = Math.min(pxPerMmX, pxPerMmY);
      const analysisScale = w / imgBitmap.width;

      const p = getPreset();
      const minStrokePx = p.minStrokeMm * pxPerMm;
      const minGapPx = p.minGapMm * pxPerMm;

      const bin = autoBinarizeInk(imgData);
      const ink = bin.inkMask;

      const distInInkSq = edt2dFromMask(ink, w, h, true); // feature: background (ink=0)
      const inv = invertMask(ink);
      const distInBgSq = edt2dFromMask(inv, w, h, true);  // feature: ink (inv=0)

      const strokeR2 = (minStrokePx * 0.5) ** 2;
      const gapR2 = (minGapPx * 0.5) ** 2;

      const thinCenters = new Uint8Array(w * h);
      const gapCenters = new Uint8Array(w * h);

      for (let y = 1; y < h - 1; y++) {
        const row = y * w;
        for (let x = 1; x < w - 1; x++) {
          const i = row + x;

          if (ink[i] === 1) {
            const d = distInInkSq[i];
            if (d > 0 && d <= strokeR2 && isLocalMaxSq(distInInkSq, ink, w, h, x, y, 1)) {
              thinCenters[i] = 1;
            }
          } else {
            const d = distInBgSq[i];
            if (d > 0 && d <= gapR2 && isLocalMaxSq(distInBgSq, ink, w, h, x, y, 0)) {
              gapCenters[i] = 1;
            }
          }
        }
      }

      const thin = dilate(thinCenters, w, h, p.dilatePx);
      const gap = dilate(gapCenters, w, h, p.dilatePx);

      const overlay = ctxHl.createImageData(w, h);
      const dd = overlay.data;

      let outThin = 0, outGap = 0;
      for (let i = 0; i < w * h; i++) {
        const k = i * 4;
        if (thin[i]) {
          dd[k] = 255; dd[k + 1] = 59; dd[k + 2] = 48; dd[k + 3] = 220;
          outThin++;
        } else if (gap[i]) {
          dd[k] = 255; dd[k + 1] = 179; dd[k + 2] = 0; dd[k + 3] = 220;
          outGap++;
        }
      }

      lastW = w; lastH = h;
      lastOverlay = overlay;
      analyzed = true;
      updateAnalyzeEnabled();

      renderOverlay();

      const polLabel =
        (bin.polarity === "alpha") ? "Transparent = background (alpha)" :
          (bin.polarity === "dark" ? "Dark = ink" : "Light = ink");
      const thrLabel = bin.threshold;
      const profLabel = (PRINT_PROFILES[printProfile] || PRINT_PROFILES.white).label;

      const inchW = mm.w / 25.4;
      const inchH = mm.h / 25.4;
      const ppiX = w / inchW;
      const ppiY = h / inchH;
      const stressLabel = stressPPI ? String(stressPPI) : "OFF";

      const inkCov = (typeof bin.coverage === "number") ? (bin.coverage * 100).toFixed(2) : "n/a";
      const extraHint = ((outThin + outGap) === 0)
        ? " / 0 hits → ink detection may be off (for transparent PNG: keep transparency; or strokes may already be thick enough)"
        : "";

      setStatus(
        `Done: thin strokes ${outThin.toLocaleString()} px / narrow gaps ${outGap.toLocaleString()} px ` +
        `(${mode === "std" ? "Standard" : "Stricter"} / Print ${profLabel} / Min stroke ${p.minStrokeMm}mm / Min gap ${p.minGapMm}mm` +
        ` / Effective PPI ${ppiX.toFixed(1)}×${ppiY.toFixed(1)} / Stress ${stressLabel}` +
        ` / Threshold ${thrLabel} / ${polLabel} / Ink coverage ${inkCov}% / Analysis scale ${analysisScale.toFixed(2)}${extraHint})`
      );

    } catch (err) {
      console.error(err);
      setStatus("Analysis failed. If the image is too large, try a smaller PNG and re-run.");
    } finally {
      btnAnalyze.disabled = false;
    }
  });

  function renderOverlay() {
    if (!imgBitmap) return;
    if (!analyzed) return;

    ctxRes.clearRect(0, 0, cRes.width, cRes.height);
    ctxRes.drawImage(imgBitmap, 0, 0, cRes.width, cRes.height);

    ctxHl.clearRect(0, 0, cHl.width, cHl.height);
    if (!elHL.checked) return;
    if (!lastOverlay) return;
    if (cHl.width !== lastW || cHl.height !== lastH) return;
    ctxHl.putImageData(lastOverlay, 0, 0);
  }

  // ---------- binarization ----------
  function autoBinarizeInk(imgData) {
    const d = imgData.data;
    const w = imgData.width;
    const h = imgData.height;
    const n = w * h;

    let transparent = 0;
    for (let i = 0; i < d.length; i += 4) {
      if (d[i + 3] === 0) transparent++;
    }
    const transparentRatio = transparent / n;

    // Transparent PNG: alpha mask is the most stable
    if (transparentRatio >= 0.03) {
      const ALPHA_THR = 16;
      const inkMask = new Uint8Array(n);
      let inkCount = 0;
      for (let p = 0, i = 0; p < n; p++, i += 4) {
        const a = d[i + 3];
        const v = (a >= ALPHA_THR) ? 1 : 0;
        inkMask[p] = v;
        if (v) inkCount++;
      }
      return {
        inkMask,
        threshold: `alpha>=${ALPHA_THR}`,
        polarity: "alpha",
        coverage: inkCount / n
      };
    }

    const candDark = buildInknessAndOtsu(imgData, "dark");
    const candLight = buildInknessAndOtsu(imgData, "light");
    let pick = chooseBetterCandidate(candDark, candLight);

    if (pick.coverage < 0.001 || pick.coverage > 0.95) {
      const alt = (pick === candDark) ? candLight : candDark;
      if (alt.coverage >= 0.001 && alt.coverage <= 0.95) pick = alt;
    }

    const { threshold, polarity, inkness } = pick;
    const inkMask = new Uint8Array(n);
    const thr = Math.max(1, Math.min(254, threshold));
    for (let i = 0; i < n; i++) inkMask[i] = (inkness[i] >= thr) ? 1 : 0;

    return { inkMask, threshold: thr, polarity, coverage: pick.coverage };
  }

  function buildInknessAndOtsu(imgData, polarity) {
    const d = imgData.data;
    const w = imgData.width;
    const h = imgData.height;

    const inkness = new Uint8Array(w * h);
    const hist = new Uint32Array(256);

    for (let i = 0, p = 0; i < d.length; i += 4, p++) {
      const r = d[i], g = d[i + 1], b = d[i + 2], a = d[i + 3];
      if (a === 0) {
        inkness[p] = 0;
        hist[0]++;
        continue;
      }
      const l = 0.2126 * r + 0.7152 * g + 0.0722 * b;
      const base = (polarity === "light") ? l : (255 - l);
      const ink = (base * a) / 255;
      const v = clamp255(Math.round(ink));
      inkness[p] = v;
      hist[v]++;
    }

    const threshold = otsuThreshold(hist);
    const stats = coverageAndVariance(hist, threshold);
    return { w, h, polarity, threshold, inkness, ...stats };
  }

  function chooseBetterCandidate(a, b) {
    const score = (c) => {
      const cov = c.coverage;
      const inRange = (cov >= 0.005 && cov <= 0.75) ? 1 : 0.15;
      return c.betweenVar * inRange;
    };
    return score(a) >= score(b) ? a : b;
  }

  function clamp255(v) { return v < 0 ? 0 : (v > 255 ? 255 : v); }

  function otsuThreshold(hist) {
    let total = 0;
    let sum = 0;
    for (let i = 0; i < 256; i++) { total += hist[i]; sum += i * hist[i]; }
    if (total === 0) return 128;

    let sumB = 0;
    let wB = 0;
    let maxVar = -1;
    let bestT = 128;

    for (let t = 0; t < 256; t++) {
      wB += hist[t];
      if (wB === 0) continue;
      const wF = total - wB;
      if (wF === 0) break;

      sumB += t * hist[t];
      const mB = sumB / wB;
      const mF = (sum - sumB) / wF;

      const between = wB * wF * (mB - mF) * (mB - mF);
      if (between > maxVar) { maxVar = between; bestT = t; }
    }
    return bestT;
  }

  function coverageAndVariance(hist, threshold) {
    let total = 0, ink = 0;
    for (let i = 0; i < 256; i++) {
      total += hist[i];
      if (i >= threshold) ink += hist[i];
    }
    const coverage = total ? (ink / total) : 0;

    let sum = 0;
    for (let i = 0; i < 256; i++) sum += i * hist[i];

    let sumB = 0;
    let wB = 0;
    let maxVar = 0;
    for (let t = 0; t < 256; t++) {
      wB += hist[t];
      if (wB === 0) continue;
      const wF = total - wB;
      if (wF === 0) break;
      sumB += t * hist[t];
      const mB = sumB / wB;
      const mF = (sum - sumB) / wF;
      const between = wB * wF * (mB - mF) * (mB - mF);
      if (between > maxVar) maxVar = between;
    }
    const betweenVar = total ? (maxVar / (total * total * 255 * 255)) : 0;
    return { coverage, betweenVar };
  }

  // ---------- EDT ----------
  const INF = 1e20;

  function invertMask(mask) {
    const out = new Uint8Array(mask.length);
    for (let i = 0; i < mask.length; i++) out[i] = mask[i] ? 0 : 1;
    return out;
  }

  function edt2dFromMask(mask, w, h, featureIsZero) {
    const f = new Float64Array(w * h);
    for (let i = 0; i < w * h; i++) {
      const isFeature = featureIsZero ? (mask[i] === 0) : (mask[i] === 1);
      f[i] = isFeature ? 0 : INF;
    }

    const g = new Float64Array(w * h);
    const tmp = new Float64Array(Math.max(w, h));
    const out = new Float64Array(Math.max(w, h));

    for (let y = 0; y < h; y++) {
      const base = y * w;
      for (let x = 0; x < w; x++) tmp[x] = f[base + x];
      edt1d(tmp, w, out);
      for (let x = 0; x < w; x++) g[base + x] = out[x];
    }

    const d = new Float64Array(w * h);
    for (let x = 0; x < w; x++) {
      for (let y = 0; y < h; y++) tmp[y] = g[y * w + x];
      edt1d(tmp, h, out);
      for (let y = 0; y < h; y++) d[y * w + x] = out[y];
    }

    return d;
  }

  function edt1d(f, n, out) {
    const v = new Int32Array(n);
    const z = new Float64Array(n + 1);

    let k = 0;
    v[0] = 0;
    z[0] = -INF;
    z[1] = INF;

    for (let q = 1; q < n; q++) {
      let s;
      while (true) {
        const vk = v[k];
        s = ((f[q] + q * q) - (f[vk] + vk * vk)) / (2 * q - 2 * vk);
        if (s > z[k]) break;
        k--;
        if (k < 0) { k = 0; break; }
      }
      k++;
      v[k] = q;
      z[k] = s;
      z[k + 1] = INF;
    }

    k = 0;
    for (let q = 0; q < n; q++) {
      while (z[k + 1] < q) k++;
      const vk = v[k];
      const dx = q - vk;
      out[q] = dx * dx + f[vk];
    }
  }

  // ---------- local maxima + dilation ----------
  function isLocalMaxSq(distSq, regionMask, w, h, x, y, regionVal) {
    const i = y * w + x;
    const v = distSq[i];
    for (let dy = -1; dy <= 1; dy++) {
      for (let dx = -1; dx <= 1; dx++) {
        if (dx === 0 && dy === 0) continue;
        const nx = x + dx;
        const ny = y + dy;
        const ni = ny * w + nx;
        if (regionMask[ni] !== regionVal) continue;
        if (distSq[ni] > v) return false;
      }
    }
    return true;
  }

  function dilate(mask, w, h, r) {
    if (r <= 0) return mask;
    const out = new Uint8Array(mask.length);
    const r2 = r * r;

    for (let y = 0; y < h; y++) {
      const row = y * w;
      for (let x = 0; x < w; x++) {
        const i = row + x;
        if (!mask[i]) continue;

        for (let dy = -r; dy <= r; dy++) {
          const ny = y + dy;
          if (ny < 0 || ny >= h) continue;
          const nrow = ny * w;
          for (let dx = -r; dx <= r; dx++) {
            if (dx * dx + dy * dy > r2) continue;
            const nx = x + dx;
            if (nx < 0 || nx >= w) continue;
            out[nrow + nx] = 1;
          }
        }
      }
    }
    return out;
  }

  // ---------- download ----------
  document.querySelectorAll("button.dl").forEach(btn => {
    btn.addEventListener("click", () => {
      if (!imgBitmap) return;

      const target = btn.dataset.target;
      if (target === "orig") {
        saveCanvas(cOrig, `${imgName}_original.png`);
        return;
      }

      const tmp = document.createElement("canvas");
      tmp.width = cRes.width;
      tmp.height = cRes.height;
      const tctx = tmp.getContext("2d");

      const matte = (previewBgMode === "light") ? "#ffffff" : "#0b0f16";
      tctx.fillStyle = matte;
      tctx.fillRect(0, 0, tmp.width, tmp.height);

      tctx.drawImage(cRes, 0, 0);
      if (elHL.checked) tctx.drawImage(cHl, 0, 0);
      saveCanvas(tmp, `${imgName}_risk-highlight.png`);
    });
  });

  function saveCanvas(canvas, filename) {
    canvas.toBlob((blob) => {
      if (!blob) return;
      const a = document.createElement("a");
      a.href = URL.createObjectURL(blob);
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      a.remove();
      setTimeout(() => URL.revokeObjectURL(a.href), 2000);
    }, "image/png");
  }

})();
